/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.ComponentModel;
using System.Drawing;
using System.Globalization;

using Borland.Eco.UmlRt;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.Services;
using Borland.Eco.Subscription;
using Borland.Eco.Globalization;

namespace Borland.Eco.Handles
{
	// Item
	[Serializable()]
	[TypeConverter("Borland.Eco.Handles.Design.OclVariableConverter, Borland.Eco.Handles.Design")]
	public sealed class OclVariable
	{
		internal bool LinksToHandle(ElementHandle handle)
		{
			bool result = (ElementHandle != null) && (handle == ElementHandle);
			if (!result && (ElementHandle is RootedHandle))
			{
				result = (ElementHandle as RootedHandle).IsRootLinkedTo(handle);
			}
			return result;
		}

		[NonSerialized()]
		private VariableDefinition variableDefinition;
		[NonSerialized()]
		private OclVariableCollection m_Owner;
		internal void SetOwner(OclVariableCollection value)
		{
			m_Owner = value;
		}

		public OclVariable()
		{
			SetDefinition(string.Empty, null);
		}

		public OclVariable(string variableName, ElementHandle elementHandle)
		{
			SetDefinition(variableName, elementHandle);
		}

		public OclVariable(OclVariable oclVariable) // Clone
		{
			variableDefinition = oclVariable.variableDefinition;
		}

		private void SetDefinition(string variableName, ElementHandle elementHandle)
		{
			variableDefinition = new VariableDefinition(variableName, elementHandle);
			if (m_Owner != null)
				m_Owner.Changed();
		}

		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		internal IExternalVariable Variable
		{
			get { return variableDefinition; }
		}

		public override string ToString()
		{
			if ((variableDefinition.ElementHandle != null) &&
				(((IComponent)variableDefinition.ElementHandle).Site != null))
				return variableDefinition.Name + ": " + ((IComponent)variableDefinition.ElementHandle).Site.Name;
			else
				return variableDefinition.Name + ": " + HandlesStringRes.sNotConnected;
		}
		private void CheckUniqueName(string name)
		{
			if (m_Owner == null) return;
			if (!m_Owner.NameIsUnique(name))
				throw new InvalidOperationException(HandlesStringRes.sNameNotUnique(name));			
		}
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryOCL")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyVariableName")]
		public string VariableName
		{
			get { return variableDefinition.Name; }
			set 
			{ 
				CheckUniqueName(value);
				SetDefinition(value, variableDefinition.ElementHandle); 
			}
		}

		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryConnections")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyElementHandle")]
		public ElementHandle ElementHandle
		{
			get { return variableDefinition.ElementHandle; }
			set { SetDefinition(variableDefinition.Name, value); }
		}

	// The immutable variable
		private sealed class VariableDefinition: IExternalVariable
		{
			private ElementHandle elementHandle;
			IElement IExternalVariable.Element
			{
				get { return elementHandle.Element; }
			}

			public IStaticContext StaticContext
			{
				get { return elementHandle; }
			}

			void IExternalVariable.Subscribe(ISubscriber subscriber)
			{
				if (elementHandle != null)
					elementHandle.SubscribeToElement(subscriber);
			}

			IClassifier IExternalVariable.StaticUmlType
			{
				get
				{
					if (StaticContext == null)
						return null;
					else
						return StaticContext.StaticUmlType;
				}
			}

			private String variableName;
			public String Name
			{
				get { return variableName; }
			}

			public VariableDefinition(string aVariableName, ElementHandle aHandle)
			{
				variableName = aVariableName;
				elementHandle = aHandle;
			}

			public ElementHandle ElementHandle
			{
				get { return elementHandle; }
			}
		}
	}

/*	internal class ExternalVariableListAdapter: IExternalVariableList
	{
		OclVariableCollection adaptee;
		internal ExternalVariableListAdapter(OclVariableCollection adaptee)
		{
			this.adaptee = adaptee;
		}

		int IExternalVariableList.Count { get { return adaptee.Count; } }
		IExternalVariable IExternalVariableList.this[int index] { get { return adaptee[index].Variable; } }
		IExternalVariable IExternalVariableList.this[string name] { get { return adaptee[name]; } }
		void IExternalVariableList.Subscribe(ISubscriber subscriber) { adaptee.Subscribe(subscriber); }
		IEnumerator IEnumerable.GetEnumerator()
		{
			ArrayList list = new ArrayList();
			foreach (OclVariable v in adaptee)
				list.Add(v.Variable);
			return list.GetEnumerator();
		}
	}
	*/

	public sealed class OclVariableCollection: System.Collections.CollectionBase, IExternalVariableList
	{

		private sealed class ElementSubscriberAdapter: SubscriberAdapterBase
		{
			protected override void DoReceive(object sender, EventArgs e, object actualSubscriber)
			{
				if (actualSubscriber == null) throw new ArgumentNullException("actualSubscriber"); // do not localize
				((OclVariableCollection)actualSubscriber).Changed();
			}

			public ElementSubscriberAdapter(object subscriber): base(subscriber) {}
		}

		IExternalVariable IExternalVariableList.this[int index]
		{
			get { return this[index].Variable; }
		}

		IEnumerator IExternalVariableList.GetEnumerator()
		{
			ArrayList list = new ArrayList();
			foreach (OclVariable v in this)
				list.Add(v.Variable);
			return list.GetEnumerator();
		}

//		public IExternalVariableList ExternalVariableList { get { return new ExternalVariableListAdapter(this); } }

		IExternalVariable IExternalVariableList.this[string name]
		{
			get
			{
				for (int i = 0; i < Count; i++)
					if (this[i].Variable.Name == name)
						return this[i].Variable;
				return null;
			}
		}

		void IExternalVariableList.Subscribe(ISubscriber subscriber)
		{
			publisher.AddSubscription(subscriber);
		}

		private Publisher publisher = new Publisher();

		public OclVariableCollection()
		{
			subscriber = new ElementSubscriberAdapter(this);
		}

		private ElementSubscriberAdapter subscriber;
		internal void Changed()
		{
			publisher.Send(this, EventArgs.Empty);
			// Always resubscribe on changed, even though not strictly needed on type change
			subscriber.Deactivate();
			subscriber = new ElementSubscriberAdapter(this);
			for (int i = 0; i < Count; i++)
			{
				ElementHandle eh = this[i].ElementHandle;
				if (eh != null)
					eh.SubscribeToStaticContext(subscriber);
			}
		}

		internal Boolean NameIsUnique(String name)
		{
			for (int i = 0; i < Count; i++)
				if (string.Compare(name, this[i].VariableName, true, CultureInfo.InvariantCulture) == 0) // case insensitive
					return false;
			return true;
		}

		private String StringGetUniqueName()
		{
			int i = 1;
			String value;
			String rootName = HandlesStringRes.sVariableRoot;

			do
			{
				value = rootName + i.ToString(CultureInfo.InvariantCulture);
				i++;
			} while (!NameIsUnique(value));
			return value;
		}

		public OclVariable this[int index]
		{
			get { return (OclVariable) List[index]; }
		}

		///<exception cref="InvalidOperationException">Thrown if the inserted OclVariable's name is not unique in the owning collection.</exception>
		protected override void OnInsert(int index, object value)
		{
			OclVariable aOclVariable = (OclVariable)value;

			aOclVariable.SetOwner(this);
			if (aOclVariable.VariableName == null || aOclVariable.VariableName.Length == 0)
				aOclVariable.VariableName = StringGetUniqueName();
			if (!NameIsUnique(aOclVariable.VariableName))
				throw new InvalidOperationException(HandlesStringRes.sNameNotUnique(aOclVariable.VariableName));
				
			base.OnInsert(index, value);
			Changed();
		}

		public void Add(OclVariable oclVariable)
		{
			List.Add(oclVariable);
		}

		public void AddRange(OclVariable[] oclVariables)
		{
			foreach(OclVariable aOclVariable in oclVariables)
				Add(aOclVariable);
		}

		///<exception cref="ArgumentNullException">Thrown if <paramref name="baseVars"/> is null.</exception>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="overrideVars"/> is null.</exception>
		static public OclVariableCollection Merge(OclVariableCollection baseVars, OclVariableCollection overrideVars)
		{
			if (baseVars == null) throw new ArgumentNullException("baseVars"); // do not localize
			if (overrideVars == null) throw new ArgumentNullException("overrideVars"); // do not localize
			OclVariableCollection newList = new OclVariableCollection();

			for (int i = 0; i < baseVars.Count; i++)
			{
				if (overrideVars.NameIsUnique(baseVars[i].VariableName))
					newList.Add(new OclVariable(baseVars[i]));
			}

			for (int i = 0; i < overrideVars.Count; i++)
				newList.Add(new OclVariable(overrideVars[i]));
			return newList;
		}
	}

	[ToolboxBitmap(typeof(OclVariables), "Borland.Eco.Handles.OclVariables.bmp")]
	public sealed class OclVariables: System.ComponentModel.Component
	{
		internal bool LinksToHandle(ElementHandle handle)
		{
			bool result = false;
			for (int i = 0; i < oclVariableCollection.Count; i++)
				result = result || oclVariableCollection[i].LinksToHandle(handle);
			return result;
		}

		public OclVariables() {}

		private OclVariableCollection oclVariableCollection = new OclVariableCollection();
		[Editor("CollectionEditor", "UITypeEditor")]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Content)]
		[LocalizableCategory(typeof(HandlesStringRes), "sCategoryOCL")]
		[LocalizableDescription(typeof(HandlesStringRes), "sPropertyOclVariableCollection")]
		public OclVariableCollection OclVariableCollection
		{
			get { return oclVariableCollection; }
		}
	}
}
